#!/usr/bin/python3

import pandas as pd
import matplotlib.pyplot as plt
import argparse
import sys
import os
import json
import logging
from copy import deepcopy


def read_next_epoch(fp, one_sec=False):
    line = ""
    if one_sec:
        while not('[HRPROC]' in line and
                  'MetaData::SatUsage' in line and
                  '.000' in line):
            line = fp.readline()
            if not line:
                logging.info('end of file')
                return None, None
    else:
        while not('[HRPROC]' in line and
                  'MetaData::SatUsage' in line):
            line = fp.readline()
            if not line:
                logging.info('end of file')
                return None, None
    epoch = float(line.split()[1])
    logging.info('--------------------')
    logging.info(f'Found epoch {epoch}')

    data = {}
    while True:
        last_pos = fp.tell()
        line = fp.readline()
        if not line:
            logging.error('Unexpected end of file')
            return None, None

        # Data lines look like L1C [x]..., L8X [x]..., etc
        if not line.startswith('L'):
            break
        if not line[1].isnumeric():
            break
        if line[4] != '[' and line[6] != ']':
            break
        if 'MetaData::SatUsage' in line:
            fp.seek(last_pos)
            break
        #print(line)
        freq = line.split()[0]
        #space = line.find(' ')
        sys_line = line[4:].rstrip()
        d_sys = {}
        for sp in sys_line.split():
            d_sys[sp[1]] = sp[3:]
        #data[freq] = line[4:].rstrip()
        data[freq] = d_sys
        logging.info(freq, data[freq])
        #data = data + line.rstrip() + ' '
    #print(f'{data=}')
    return epoch, data


def diff_letters(a,b):
    return sum ( a[i] != b[i] for i in range(len(a)) )

def num_sats_used(d):
    num_sat_changes = 0
    for d_sys, d_sats in d.items():
        logging.info(f'      leftover sys {d_sys}: {d_sats}')
        n = d_sats.count('+')
        num_sat_changes += n
        logging.info(f'      {n=}, {num_sat_changes=}')
    return num_sat_changes


def diff_sys(a, b):
    num_sat_changes = 0
    for a_sys in list(a):
        a_sats = a[a_sys]
        for b_sys in list(b):
            b_sats = b[b_sys]
            if a_sys == b_sys:
                logging.info(f'   Found sys {a_sys} in both')
                logging.info(f'      {a_sats}')
                logging.info(f'      {b_sats}')
                n = diff_letters(a_sats, b_sats)
                num_sat_changes += n
                logging.info(f'      {n=}, {num_sat_changes=}')
                del a[a_sys]
                del b[b_sys]
                break

    logging.info(f'   leftover sys in a:')
    n = num_sats_used(a)
    num_sat_changes += n
    logging.info(f'      {n=}, {num_sat_changes=}')

    logging.info(f'   leftover sys in b:')
    n = num_sats_used(b)
    num_sat_changes += n
    logging.info(f'      {n=}, {num_sat_changes=}')

    return num_sat_changes


def diff_sats(a, b):
    a = deepcopy(a)
    b = deepcopy(b)
    num_changes = 0
    for a_band in list(a):
        a_sys = a[a_band]
        for b_band in list(b):
            b_sys = b[b_band]
            if a_band == b_band:
                logging.info(f'Found band {a_band} in both')
                logging.info(f'   {a_sys}')
                logging.info(f'   {b_sys}')
                n = diff_sys(a_sys, b_sys)
                num_changes += n
                logging.info(f'   {n=}, {num_changes=}')
                del a[a_band]
                del b[b_band]
                break

    for a_band, a_sys in a.items():
        logging.info(f'   leftover band {a_band} in a')
        n = num_sats_used(a_sys)
        num_changes += n
        logging.info(f'   {n=}, {num_changes=}')

    for b_band, b_sys in b.items():
        logging.info(f'   leftover band {b_band} in b')
        n = num_sats_used(b_sys)
        num_changes += n
        logging.info(f'   {n=}, {num_changes=}')
    #print(f'{a=}')
    #print(f'{b=}')
    return num_changes


def calc_changes_and_plot(filename, one_sec, plot=True):
    epoch = 1
    sat_changes = []
    readA = True
    epochA_sats = None
                #while lineB := fileB.readline():
    fp = open(filename)

    epoch_prev, epoch_sats_prev = read_next_epoch(fp, one_sec)
    #print(f'{epoch_prev=}')
    #print(f'{len(epoch_sats_prev)=}')
    #print('epoch_sats_prev:')
    #print(json.dumps(epoch_sats_prev, indent=4))
    while epoch:
        epoch, epoch_sats = read_next_epoch(fp, one_sec)
        if epoch is None:
            break

        n = diff_sats(epoch_sats_prev, epoch_sats)
        #print(epoch, n)
        sat_changes.append({'Time': epoch, 'sat_changes': n})

        epoch_sats_prev = epoch_sats
        #sys.exit()
        #input()
    #print(df)
    fp.close()

    fname_out = f'{filename[:-4]}_sat_changes.csv'
    print(f'Saving {fname_out}')
    df = pd.DataFrame(sat_changes)
    df.to_csv(fname_out, index=False)

    if plot:
        fig = plt.figure()
        plt.plot(df.Time, df.sat_changes, '.')
        plt.title(filename)
        plt.xlabel('Time (s)')
        plt.ylabel('Num Sats Changed')

        png_fname = f'{filename[:-4]}_sat_changes.png'
        print(f'Saving {png_fname}')
        plt.savefig(png_fname)

        plt.close(fig)

    return df


def main():
    parser = argparse.ArgumentParser(
        description='Apply an elevation mask with t0x2t0x.',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
    )
    parser.add_argument(
        '-f', "--filename",
        default='TitanPlayer_log.txt',
        help="File to process"
    )
    #parser.add_argument(
    #    "-n", "--dry_run",
    #    action='store_true',
    #    help="Don't actually run t0x2t0x, just print the commands."
    #)
    parser.add_argument(
        "-1", "--one_sec",
        action='store_true',
        help="only process precise solutions that are on one second boundaries."
    )
    parser.add_argument(
        "--plot",
        action=argparse.BooleanOptionalAction,
        default=True,
        help="Create plots"
    )
    parser.add_argument(
        "--verbose", "-v",
        action='count',
        default=0,
        help="use multiple -v for more detailed messages."
    )

    logger_format = (
            #"[%(asctime)s][%(levelname)-8s]"
            "[%(levelname)-8s]"
            "[%(filename)11s:%(lineno)-5s] "
            "%(funcName)-10s: %(message)s"
            )
    args = parser.parse_args()
    logging.basicConfig(
        level=logging.WARNING - (10 * args.verbose),
        format=logger_format,
        stream=sys.stdout
    )
    logging.debug(args)
    calc_changes_and_plot(args.filename, args.one_sec, args.plot)


if __name__ == '__main__':
    main()
