#!/usr/bin/python3

import pandas as pd
import matplotlib.pyplot as plt
import argparse
import sys
import os
import logging

from plot_sat_set_change import calc_changes_and_plot

def main():
    parser = argparse.ArgumentParser(
        description='Plot sat changes in FOLDER/outputs/titan/DEV/',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
    )
    parser.add_argument(
        '-F', '--folder',
        default='.',
        help='Root dir containing T04 Files.',
        )
    parser.add_argument(
        '-D', '--devs',
        type=str,
        nargs='+',
        default=['BX992', 'EB1-5093', 'EB1-5095'], #, 'EB-5081', 'P4-05'],
        help="List of dev directories, will be used as FOLDER/day/data/gnss/DEV"
             "If None, then scan FOLDER/day/data/gnss/ for dev directories that "
             "do not start with '.' or '_'. "
             "Example: -D BXLVL JudoRoof P4-05 MS956",
        )
    parser.add_argument(
        "-1", "--one_sec",
        action='store_true',
        help="only process precise solutions that are on one second boundaries."
    )
    parser.add_argument(
        "--plot-individual",
        action=argparse.BooleanOptionalAction,
        default=False,
        help="Create plots"
    )
    parser.add_argument(
        "--force",
        action=argparse.BooleanOptionalAction,
        default=False,
        help="Run the counting algorithm, even if the .csv exists"
    )
    parser.add_argument(
        "--verbose", "-v",
        action='count',
        default=0,
        help="use multiple -v for more detailed messages."
    )

    logger_format = (
            #"[%(asctime)s][%(levelname)-8s]"
            "[%(levelname)-8s]"
            "[%(filename)11s:%(lineno)-5s] "
            "%(funcName)-10s: %(message)s"
            )
    args = parser.parse_args()
    logging.basicConfig(
        level=logging.WARNING - (10 * args.verbose),
        format=logger_format,
        stream=sys.stdout
    )
    logging.debug(args)

    #df_list = []
    fig = plt.figure(figsize=(20, 5), layout="constrained")
    ax = plt.gca()
    for dev in args.devs:
        csv_fname = os.path.join(args.folder, 'outputs', 'titan',
                                dev, f'TitanPlayer_{dev}_sat_changes.csv')
        filename = os.path.join(args.folder, 'outputs', 'titan',
                                dev, f'TitanPlayer_{dev}.log')
        if os.path.isfile(csv_fname) and not args.force:
            df = pd.read_csv(csv_fname)
        else:
            df = calc_changes_and_plot(filename, args.one_sec, args.plot_individual)
        ax.plot(df.Time, df.sat_changes,
                '.',
                markersize=4,
                label=dev)

    plt.title(f'{args.folder} Number of Satellite Changes')
    #plt.legend(args.devs)
    #ax.legend(bbox_to_anchor=(1.1, 1.05))
    ax.legend(bbox_to_anchor=(1.04, 0.5), loc="center left", borderaxespad=0,
              markerscale=4.,
              )
    plt.xlabel('Time (s)')
    plt.ylabel('Num Sats Changed')
    plt.grid(True)

    png_fname = os.path.join(args.folder, f'sat_changes.png')
    print(f'Saving {png_fname}')
    plt.savefig(png_fname)
    plt.show()
    plt.close()

if __name__ == '__main__':
    main()
