#!/usr/bin/python3

import os
import subprocess
import time
import pandas as pd
import sys
import argparse
import logging


def main():
    parser = argparse.ArgumentParser(
        description='Run TitanPlayer on FOLDER/*/data/gnss/*/dut_dir',
        epilog="Skip folders starting with '.' or '_'",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
        )
    parser.add_argument(
        '--folder',
        default='.',
        help='Root dir containing FOLDERS.',
        )
    parser.add_argument(
        '-a', '--ant',
        choices=['0', '1'],
        help='Antenna number',
        )
    parser.add_argument(
        '--dut_dirs',
        nargs='+',
        #default=['P3ENCL'],
        default=['P4-03', 'P4-05'],
        help='Pass FOLDER/*/data/gnss/dut_dir/*.T0[24] to TitanPlayer',
        )
    debugp = parser.add_argument_group('Debug options')
    debugp.add_argument(
        "--dryrun",
        action='store_true',
        help="Don't actually run TitanPlayer",
        )
    debugp.add_argument(
        '-1', "--do_one",
        action='store_true',
        help="Quit after the first TitanPlayer call",
        )
    debugp.add_argument(
        "--verbose", "-v",
        action='count',
        default=0,
        help="use multiple -v for more detailed messages."
    )
    args = parser.parse_args()
    folder = os.path.abspath(args.folder)

    logger_format = ("[%(levelname)-8s]"
                     "[%(filename)11s:%(lineno)-5s] "
                     "%(funcName)-10s: %(message)s"
                     )
    logging.basicConfig(
        level=logging.WARNING - (10 * args.verbose),
        format=logger_format,
        stream=sys.stdout
    )

    with os.scandir(folder) as it:
        for entry in it:
            d = entry.name
            if not entry.is_dir() or d.startswith('_') or d.startswith('.'):
                continue
            print(f"============ {d} =====================")
            for dut in args.dut_dirs:
                print(f"   -------- {dut} ----------")
                din = os.path.join(folder, d, 'data', 'gnss', dut)
                dout = os.path.join(folder, d, 'gen_data', f'ant{args.ant}', dut)
                logdir = os.path.join(folder, d, 'logs', f'ant{args.ant}', dut)
                os.makedirs(dout, exist_ok=True)
                os.makedirs(logdir, exist_ok=True)

                with os.scandir(din) as t04_it:
                    for entry in t04_it:
                        fn_t04 = entry.name
                        log_name = os.path.join(logdir, f't0x2t0x_{fn_t04}.log')
                        path_t04 = os.path.join(din, fn_t04)
                        if not os.path.isfile(path_t04) or fn_t04.startswith(','):
                            continue
                        if not (fn_t04.endswith('.T04') or fn_t04.endswith('T02')):
                            continue
                        logging.debug(f'   {fn_t04}')
                        cmd = ['t0x2t0x', f'-a{args.ant}',
                                path_t04,
                                os.path.join(dout, fn_t04),
                                ]
                        logging.info(' '.join(cmd))
                        if not args.dryrun:
                            flog = open(log_name, 'w')
                            p = subprocess.run(
                                cmd, stderr=subprocess.STDOUT, stdout=flog
                            )
                            flog.close()
                            if p.returncode != 0:
                                logging.warning(f't0x2t0x return code {p.returncode}')
                        if args.do_one:
                            sys.exit()


if __name__ == '__main__':
    main()
