import argparse
import subprocess
import os
import glob


def main():
    parser = argparse.ArgumentParser(
        description='Process long runs of T04 data, where mutiple (livesky) '
        'T04 files need to be combined, and then differenced. Attempt to '
        'only run the necessary processes to create any data files '
        'missing in the data pipeline.',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
        )
    parser.add_argument('-v', '--Verbose', action='count', default=0)
    parser.add_argument(
        "-r", "--ref_fname_base",
        type=str,
        default="BX",
        help="Base filename for file I/O. Inputs are .T04. "
        )
    parser.add_argument(
        "-d", "--days",
        type=str,
        nargs='+',
        default=["0115"],
        help="List of days directories"
             "Example: -d 0115 0116",
        )
    parser.add_argument(
        "-D", "--devs",
        type=str,
        nargs='+',
        default = [
            'WCO-Base', 'BXLVL', 'JudoRoof', 'survey-base',    # Bases
            'MS956', 'P4-03', 'P4-05',                         # DUTs
            ],
        help="List of dev directories, that will be inside the days dirs"
             "Example: -D BXLVL JudoRoof",
        )
    parser.add_argument(
        "--hz",
        type=float,
        default=1,
        help="Set decimation rate"
        )
    parser.add_argument(
        "-e", "--elevation",
        type=float,
        default=10,
        help="Set elevation mask."
        )
    parser.add_argument(
        "--t0x2t0x",
        default='t0x2t0x',
        help="Executable name for t0x2t0x."
        )
    parser.add_argument(
        "--log_dir", default="logs",
        help="Where to store .log files"
        )

    args = parser.parse_args()

    os.makedirs(args.log_dir, exist_ok=True)

    for day in args.days:
        for dev in args.devs:
            out_path = os.path.join(day + f"_{args.hz}Hz_{args.elevation}El", dev)
            os.makedirs(out_path, exist_ok=True)
            root_dir=os.path.join(day, dev)
            flist = sorted(glob.glob(
                            '*.T0?',
                            root_dir=root_dir,
                            ))
            #print(day, dev, root_dir, flist)
            for f in flist:
                f_in = root_dir=os.path.join(day, dev, f)
                f_out = root_dir=os.path.join(out_path, f)

                if os.path.isfile(f_out):
                    print(f"Skipping {f_out} ")
                else:
                    cmd_list = [
                        args.t0x2t0x,
                        f'-obs_dec={args.hz}',
                        f'-elev={args.elevation}',
                        f_in,
                        f_out,
                    ]
                    print(' '.join(cmd_list))
                    log_name = os.path.join(args.log_dir, f + "_t0x_decimate.log")
                    f_log = open(log_name, 'w')
                    p = subprocess.Popen(
                        cmd_list, stderr=subprocess.STDOUT, stdout=f_log
                    )
                    #processes.append((p, f_log, log_name))
                    p.wait()
                    f_log.close()
                    if p.returncode != 0:
                        print(f'WARNING: t0x2t0x return code {p.returncode}')
                    # Note: we need to scan the output from t0x2t0x because it seems
                    # to provide and exit code of 0, regardless of the error message
                    # being present or not. The typical error message is something like
                    # "T01 compression error 2: 21", which typically indicates that
                    # the t0x2t0x is older than the GNSS firmware used to collect the
                    # data.
                    with open(log_name) as f:
                        if 'error' in f.read():
                            print(f"WARNING: Found 'error' in {log_name}")

if __name__ == '__main__':
    main()
