#!/usr/bin/python3

import argparse
import subprocess
import os
import io
import sys
import pandas as pd
import shutil
import logging

def combine_hourly(hourly_dir, out_name, dryrun):
    if os.path.isfile(out_name):
        return
    print(f"combine({hourly_dir}/*.T0? to {out_name}.T0[24])")
    flist_t02 = []
    flist_t04 = []
    with os.scandir(hourly_dir) as it:
        for entry in it:
            d = entry.name
            if not entry.is_file():
                continue
            if d.startswith('_') or d.startswith('.'):
                continue
            if d == out_name:
                continue
            if d.endswith('T04'):
                flist_t04.append(os.path.join(hourly_dir, d))
            if d.endswith('T02'):
                flist_t02.append(os.path.join(hourly_dir, d))

    if flist_t02:
        flist_t02.sort()
        out_name_t02 = out_name + '.T02'
        with open(out_name_t02, 'wb') as wfd:
            for f in flist_t02:
                logging.info(f"    appending {f} to {out_name_t02}")
                if not dryrun:
                    with open(f, 'rb') as fd:
                        shutil.copyfileobj(fd, wfd)
    if flist_t04:
        flist_t04.sort()
        out_name_t04 = out_name + '.T04'
        with open(out_name_t04, 'wb') as wfd:
            for f in flist_t04:
                logging.info(f"    appending {f} to {out_name_t04}")
                if not dryrun:
                    with open(f, 'rb') as fd:
                        shutil.copyfileobj(fd, wfd)

def main():
    parser = argparse.ArgumentParser(
        description='Combine (typically hourly) T04/T02 files in a directory.',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
        epilog='Combine FOLDER/data/gnss/DEV/*.T0[24] to '
               'FOLDER/data/gnss/DEV-all/DEV_all.T0[24]. '
               'Skip folders starting with "." or "_". '
               'Deletes existing FOLDER/data/gnss/DEV-all folders first.',
        )
    parser.add_argument(
        "--folder",
        default=".",
        help="Where to scan for T04 files."
        )
    parser.add_argument(
        "-D", "--devs",
        type=str,
        nargs='+',
        default=None,
        help="List of dev directories, that will be inside the days dirs"
             "Example: -D BXLVL JudoRoof",
        )
    debugp = parser.add_argument_group('Debug options')
    debugp.add_argument(
        '-n', "--dryrun",
        action="store_true",
        help="Don't actually run t0x2t0x."
        )
    debugp.add_argument('-v', '--verbose', action='count', default=0)

    args = parser.parse_args()
    logger_format = ("[%(levelname)-8s]"
                     "[%(filename)11s:%(lineno)-5s] "
                     "%(funcName)-10s: %(message)s"
                     )
    logging.basicConfig(
        level=logging.WARNING - (10 * args.verbose),
        format=logger_format,
        stream=sys.stdout
    )
    folder = os.path.normpath(args.folder)

    with os.scandir(folder) as it:
        for entry in it:
            d = entry.name
            if not entry.is_dir() or d.startswith('_') or d.startswith('.'):
                continue
            if d == 'figures' or d == 'logs':
                continue
            print(f"============ {d} =====================")
            gnss_dir = os.path.join(folder, d, 'data', 'gnss')
            if args.devs is None:
                subfolders = [ f.name for f in os.scandir(gnss_dir) if f.is_dir() ]
            else:
                subfolders = args.devs
            subfolders.sort()
            for sf in subfolders:
                if sf.endswith('-all') or sf.endswith('_all'):
                    print(f'Deleting {os.path.join(gnss_dir, sf)}')
                    shutil.rmtree(os.path.join(gnss_dir, sf))
                    continue

            for sf in subfolders:
                if sf.endswith('-all') or sf.endswith('_all'):
                    continue
                if sf.startswith('.') or sf.startswith('_'):
                    continue
                hourly_dir = os.path.join(gnss_dir, sf)
                out_dir = os.path.join(gnss_dir, f'{sf}-all')
                out_fname = os.path.join(out_dir, f'{sf}_all')
                logging.info(f'makedirs({out_dir})')
                os.makedirs(out_dir, exist_ok=True)
                combine_hourly(hourly_dir, out_fname, args.dryrun)

if __name__ == '__main__':
    main()
